<?php

// ====================================================================================
// SHORTCODE [chat from="x" to="y"] – interfaz de conversación entre 2 usuarios
// ====================================================================================
add_shortcode("chat", function ($atts) {
    if (!is_user_logged_in()) {
        return "<div>Debes iniciar sesión para usar el chat.</div>";
    }

    $atts = shortcode_atts(array(
        "from" => 0,
        "to"   => 0,
        "post" => 0
    ), $atts, "chat");

    $from = intval($atts["from"]);
    $to   = intval($atts["to"]);
    // Resolver post_id: del shortcode o del post actual
    $post_id = intval($atts["post"]);
    if ($post_id == 0) {
        $qid = get_queried_object_id();
        if ($qid) { $post_id = intval($qid); }
    }

    if ($from == 0 || $to == 0) {
        return "<div>Invalid new chat, use [chat from=\"ID\" to=\"ID\"].</div>";
    }

    // Mostrar display names
    $u_from = get_userdata($from);
    $u_to   = get_userdata($to);

    if (!$u_from || !$u_to) {
        return "<div>No users found.</div>";
    }

    // Clave de chat
    $chat_key = esc_attr(soc_normalize_chat_key($from, $to));
// Si el hilo está reportado, mostramos solo el mensaje y salimos
if (soc_is_reported($chat_key, intval($post_id))) {
    return '<div class="soc-reported-note">Thanks for your report,<br>our team is investigating</div>';
}

    ob_start();
    ?>
    <div class="soc-chat-wrap" data-chat-key="<?php echo $chat_key; ?>" data-from="<?php echo intval($from); ?>" data-to="<?php echo intval($to); ?>" data-post="<?php echo intval($post_id); ?>">
        <!-- Encabezado con display names -->
        <div class="soc-chat-header">
            <strong>Chat:</strong>
            <span class="soc-chat-parties">
                <?php echo esc_html($u_from->display_name); ?> ↔ <?php echo esc_html($u_to->display_name); ?>
            </span>
        </div>

        <!-- Contenedor de mensajes -->
        <div class="soc-chat-messages"></div>

        <!-- Formulario de envío de mensaje -->
        <form class="soc-chat-form" enctype="multipart/form-data">
            <textarea name="soc_message" rows="3" placeholder="Message..."></textarea>
            <label for="soc_attachment" class="custom-file-label">Attach a file</label>
            <input type="file" id="soc_attachment" name="soc_attachment" class="custom-file-input" accept="*/*">
            <button type="submit">Send</button>
        </form>

    </div>

    <script>
    // Código del frontend: carga y envío de mensajes con AJAX (usa jQuery CDN)
    jQuery(document).ready(function ($) {
        // fallback local por si otro script borra window.ajaxurl
        var ajaxurl = window.ajaxurl || "<?php echo esc_url(admin_url('admin-ajax.php')); ?>";
        // Tomar elementos base
//var $wrap = $(".soc-chat-wrap[data-chat-key='<?php echo $chat_key; ?>']");
        var $wrap = $(".soc-chat-wrap[data-chat-key='<?php echo $chat_key; ?>'][data-post='<?php echo intval($post_id); ?>']");
        var chatKey = $wrap.data("chat-key");
        var fromId = parseInt($wrap.data("from"));
        var toId = parseInt($wrap.data("to"));
        var postId = parseInt($wrap.data("post")) || 0; // post_id del hilo (0 = general)
        var $msgs = $wrap.find(".soc-chat-messages");
        var $form = $wrap.find(".soc-chat-form");
        // util para escapar texto antes de imprimir en HTML
        function esc(s) {
            return $("<div>").text(s).html();
        }
        
        // estado para render incremental y control de scroll
        var lastId = 0; // último ID de mensaje renderizado
        function isNearBottom($el) {
            // true si el usuario está cerca del fondo (10px de tolerancia)
            return ($el.scrollTop() + $el.innerHeight()) >= ($el[0].scrollHeight - 10);
        }
        
        // helper para detectar si hay texto real (ignora HTML, <br>, &nbsp; y espacios)
        function hasTextContent(html){
            if (html == null) { return false; }
            // Convierte HTML a texto plano
            var txt = $("<div>").html(html).text();
            // Reemplaza &nbsp; por espacio y hace trim
            txt = $.trim(txt.replace(/\u00A0/g, " "));
            return txt.length > 0;
        }                
        // Función para formatear cada mensaje
        function renderMsg(row) {
            // Mostrar siempre display_name. Si el autor es admin, mostrar "website admin says..."
            var label = row.sender_display;
            if (row.sender_is_admin == 1) {
                label = "<span style='color: red; font-size:150%;'>Website admin</span>";
            }
            
            // determinar si el mensaje es del usuario "from"
            var isFrom = (row.sender_id == fromId); 
            var sideClass = isFrom ? "soc-from" : "soc-to"; // clase lateral
            
            // usar URL de descarga y forzar download
            var hasFile = "";
            if (row.attachment && row.attachment != "") {
                var url = row.attachment_download || row.attachment_url;
                var fileName = url.split("/").pop();
                hasFile = '<div class="soc-file"><a href="' + url + '" download target="_self">' + fileName + '</a></div>';
            }
            
            // preparar burbuja de texto solo si hay contenido real
            var msgHtml = "";
            if (hasTextContent(row.message)) { // CAMBIO: evita burbuja vacía en adjuntos sin texto
                msgHtml = '<div class="soc-text">' + row.message + '</div>';
            }
            
            var html = ''
            + '<div class="soc-msg ' + sideClass + '" >'
              +   '<div class="soc-meta" ><strong>' + label + '</strong> · ' + row.created_at + '</div>'
              +   msgHtml
              +   hasFile
              + '</div>';
            return html;
        }

        // Cargar mensajes
        // bandera para evitar llamadas simultáneas
        var isLoading = false;
        // render incremental + scroll solo si hay nuevos
        function loadMessages(forceScroll) {
            if (isLoading) { 
                return; // no hacer nada si ya hay una carga en curso
            }
            isLoading = true; // marcar carga en curso

            // detectar si el usuario ya estaba abajo
            var wasAtBottom = isNearBottom($msgs);

            $.post(ajaxurl, {
                action: "soc_fetch_messages",
                chat_key: chatKey,
                post_id: postId
            }, function (resp) {
                if (resp && resp.success == true) {
                    var list = resp.data || [];

                    // contar cuántos mensajes nuevos se agregan
                    var newCount = 0;
                    var localMax = lastId; // llevar el mayor id visto en esta pasada

                    for (var i = 0; i < list.length; i++) {
                        var row = list[i];

                        if (row.id > lastId) { // solo agregar si es nuevo
                            $msgs.append( renderMsg(row) );
                            newCount++; // contamos el nuevo
                        }

                        // mantener el mayor id observado (por seguridad)
                        if (row.id > localMax) {
                            localMax = row.id;
                        }
                    }

                    // actualizar lastId al mayor observado
                    lastId = localMax;

                    // solo hacer scroll si:
                    // - nos lo pidieron explícitamente (forceScroll), o
                    // - el usuario estaba abajo Y llegaron mensajes nuevos
                    if (forceScroll || (wasAtBottom && newCount > 0)) {
                        $msgs.scrollTop($msgs[0].scrollHeight);
                    }
                }
            }, "json").always(function () {
                isLoading = false; // liberar la bandera al terminar
            });
        }

        // Enviar mensaje (texto + adjunto)
        $form.on("submit", function (e) {
            e.preventDefault();
            
            // Validar que el mensaje no esté vacío, a menos que haya archivo
            var msg = $form.find("textarea[name='soc_message']").val();
            var fileInput = $form.find("input[name='soc_attachment']")[0];
            var hasFile = (fileInput && fileInput.files && fileInput.files.length > 0);

            // solo bloquear si NO hay texto y NO hay archivo
            if ($.trim(msg) === "" && !hasFile) {
                console.log("No puedes enviar un mensaje vacío sin adjunto");
                return; // no seguimos si no hay nada
            }
            
            var fd = new FormData();
            fd.append("action", "soc_send_message");
            fd.append("chat_key", chatKey);
            fd.append("from", fromId);
            fd.append("to", toId);
            fd.append("post_id", postId); // guardar con post_id
            fd.append("message", $form.find("textarea[name='soc_message']").val());

            var fileInput = $form.find("input[name='soc_attachment']")[0];
            if (fileInput && fileInput.files && fileInput.files.length > 0) {
                fd.append("attachment", fileInput.files[0]);
            }

            $.ajax({
                url: ajaxurl,
                method: "POST",
                data: fd,
                contentType: false,
                processData: false,
                dataType: "json",
                success: function (resp) {
                    console.log("send resp", resp);
                    if (resp && resp.success == true) {
                        $form.find("textarea[name='soc_message']").val("");
                        if (fileInput) {
                            fileInput.value = "";
                        }
                        loadMessages(true); // forzar scroll al fondo solo tras enviar
                    }
                }
            });
        });

        // Cargar al inicio
        loadMessages(true);

        // Polling simple cada cuanto refresca los chats
        setInterval(function () {
            loadMessages(false);
        }, 1000);
    });
    </script>
    <?php
    return ob_get_clean();
});

// ====================================================================================
// AJAX: enviar mensaje (usuarios)
// ====================================================================================
add_action("wp_ajax_soc_send_message", "soc_send_message");
add_action("wp_ajax_nopriv_soc_send_message", "soc_send_message");
function soc_send_message() {
    if (!is_user_logged_in()) {
        wp_send_json(array("success" => false));
    }

    global $wpdb;
    $table = $wpdb->prefix . "simple_chats";

    $chat_key = isset($_POST["chat_key"]) ? sanitize_text_field($_POST["chat_key"]) : "";
    $from     = isset($_POST["from"]) ? intval($_POST["from"]) : 0;
    $to       = isset($_POST["to"]) ? intval($_POST["to"]) : 0;
    $post_id  = isset($_POST["post_id"]) ? intval($_POST["post_id"]) : 0;
    $message  = isset($_POST["message"]) ? wp_kses_post($_POST["message"]) : "";
    // Autocompleta cuando el chat se cargó por AJAX y algún campo viene vacío.
    if ($from == 0) { $from = get_current_user_id(); } // Usa el usuario actual si falta "from".
    if ($chat_key === "" && $from > 0 && $to > 0) { $chat_key = soc_normalize_chat_key($from, $to); } // Recalcula chat_key.
    if ($chat_key === "" || $from == 0 || $to == 0) {
        wp_send_json(array("success" => false, "reason" => "missing_fields")); // Corta si aún faltan datos.
    }

    // Guardar adjunto si viene
    $attachment_rel = "";
    if (isset($_FILES["attachment"]) && isset($_FILES["attachment"]["tmp_name"]) && $_FILES["attachment"]["tmp_name"] != "") {
        $baseDir = ABSPATH . "chat-attachments";
        $name = wp_unique_filename($baseDir, basename($_FILES["attachment"]["name"]));
        $dest = $baseDir . "/" . $name;
        move_uploaded_file($_FILES["attachment"]["tmp_name"], $dest);
        $attachment_rel = site_url("/") . "chat-attachments/" . $name;
    }

    $wpdb->insert($table, array(
        "chat_key"    => $chat_key,
        "post_id"     => $post_id, 
        "sender_id"   => $from,
        "receiver_id" => $to,
        "message"     => $message,
        "attachment"  => $attachment_rel,
        "created_at"  => current_time("mysql")
    ));

    wp_send_json(array("success" => true));
}

// ====================================================================================
// AJAX: obtener mensajes por chat_key (orden viejo→nuevo)
// ====================================================================================
add_action("wp_ajax_soc_fetch_messages", "soc_fetch_messages");
add_action("wp_ajax_nopriv_soc_fetch_messages", "soc_fetch_messages");
function soc_fetch_messages() {
    global $wpdb;
    $table = $wpdb->prefix . "simple_chats";

    $chat_key = isset($_POST["chat_key"]) ? sanitize_text_field($_POST["chat_key"]) : "";
    if ($chat_key == "") {
        wp_send_json(array("success" => false, "data" => array()));
    }
    $post_id = isset($_POST["post_id"]) ? intval($_POST["post_id"]) : 0;

    if ($post_id > 0) {
        $rows = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM `$table` WHERE chat_key = %s AND post_id = %d ORDER BY created_at ASC, id ASC",
            $chat_key, $post_id
        ), ARRAY_A);
    } else {
        // Hilo "general" (post_id = 0)
        $rows = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM `$table` WHERE chat_key = %s AND post_id = 0 ORDER BY created_at ASC, id ASC",
            $chat_key
        ), ARRAY_A);
    }

    $out = array();
    if ($rows && count($rows) > 0) {
        foreach ($rows as $r) {
            $sender = get_userdata(intval($r["sender_id"]));
            $sender_display = "Usuario " . intval($r["sender_id"]);
            if ($sender) {
                $sender_display = $sender->display_name;
            }

            $sender_is_admin = 0;
            if ($sender && user_can($sender->ID, "manage_options")) {
                $sender_is_admin = 1;
            }

            $out[] = array(
                "id"               => intval($r["id"]),
                "chat_key"         => $r["chat_key"],
                "sender_id"        => intval($r["sender_id"]),
                "receiver_id"      => intval($r["receiver_id"]),
                "sender_display"   => $sender_display,
                "sender_is_admin"  => $sender_is_admin,
                "message"          => $r["message"],
                "attachment"       => $r["attachment"],
                "attachment_url"   => $r["attachment"],
                "created_at"       => $r["created_at"]
            );
        }
    }

    wp_send_json(array("success" => true, "data" => $out));
}
